/************************************************************************
* Name:         Module 014: Signal Preparation for Angular Spectrum Design
* Description:  This module compensates geometrical distortions and intensity
*               attenuations due to high NA setups. It converts the desired 
*               output far field amplitude pattern into the angular spectrum
*               for the necessary signal field.
* Authors:      Christian Hellmann (LightTrans)
* Version:      2.0 (Nov 13, 2015)
* License:      Some Rights reserved, CC-BY 3.0
*************************************************************************/


///

using System;
using System.Collections.Generic;
using System.Drawing;

using VirtualLabAPI.Core.Common;
using VirtualLabAPI.Core.FieldRepresentations;
using VirtualLabAPI.Core.Numerics;
using VirtualLabAPI.Core.Modules;
using System.Windows.Forms;
using VirtualLabAPI.UI.BasicUI;
using VirtualLabAPI.Core.SupportFunctions;

public class VLModule : IVLModule {
    public void Run() {
      //get complex amplitude to convert
      ComplexAmplitude caToConvert = null;
      int index = 0;
      caToConvert = (ComplexAmplitude)(Globals.ActiveDocumentHistory.BrowseLastDocuments(DocumentFilter.AnySpatialField,
                                                                                         out index,
                                                                                         "Select Field to Convert (Spatial Domain)"));
      //error handling
      if(caToConvert == null){
        Globals.DataDisplay.LogError("No field selected");
        return;
      }
      
      //initialize form for input of parameters
      FormSignalPreparationAngularSpectrumDesign formPreparation = new FormSignalPreparationAngularSpectrumDesign(caToConvert);
      
      if(formPreparation.ShowDialog() == DialogResult.OK){
        //get the set up parameters from the dialog
        SamplingParameters sParaConversion = formPreparation.SamplingParametersForConversion;
        Vector samplingPointsSig = sParaConversion.SamplingPoints;
        VectorD samplingDistanceSig = sParaConversion.SamplingDistance;
        double wavelength = formPreparation.Wavelength;
        double distance = formPreparation.Distance;
        bool preparationForSplitter = formPreparation.UseForBeamSplitterSignalPreparation;
        bool doEnergyCorrection = formPreparation.CorrectPowerOfOrders;
        
        //perform the convertion
        ComplexAmplitude caAngularSpectrum = CalculateSignalFieldFromTargetPlaneField(caToConvert,
                                                                                      samplingPointsSig,
                                                                                      samplingDistanceSig,
                                                                                      wavelength,
                                                                                      distance,
                                                                                      preparationForSplitter,
                                                                                      doEnergyCorrection);
        Globals.DataDisplay.ShowDocument(caAngularSpectrum, "Prepared Signal Field");
    }
  }
  
  /// <summary>
  /// [NI] Calculate approximate signal field for diffuser, beam shaper and beam splitter designs from field given in target plane.
  /// Each non-zero sampling value in both fields corresponds to a signal order of the beam splitter.
  /// </summary>
  /// <param name="targetPlaneField">Field in target plane</param>
  /// <param name="samplingPointsSig">Number of sampling points for signal field</param>
  /// <param name="samplingDistanceSig">Sampling distance for signal field in (1/m, 1/m)</param>
  /// <param name="wavelength">Wavelength to compute wave number vectors from signal field points</param>
  /// <param name="distance">Distance to origin point of target plane in (m, m, m)</param>
  /// <param name="fieldHasDiscreteOrders">If true every sampling point of the targetPlaneField is considered as a
  /// discrete diffraction order. No interpolation will be done.</param>
  /// <param name="doPowerCorrection">If true the power of the orders with higher angles will be increase to compesate an intensity reduction 
  /// because of an increasing of the laser beam diameter.</param>
  /// <returns>Aproximate signal field</returns>
  public static ComplexAmplitude CalculateSignalFieldFromTargetPlaneField(ComplexAmplitude targetPlaneField,
                                                                          Vector samplingPointsSig,
                                                                          VectorD samplingDistanceSig,
                                                                          double wavelength,
                                                                          double distance,
                                                                          bool fieldHasDiscreteOrders,
                                                                          bool doPowerCorrection) {
    ComplexAmplitude signalField;

    if (fieldHasDiscreteOrders) {
      signalField = CalculateSignalFieldFromTargetPlaneFieldBeamSplitter(targetPlaneField,
                                                                         samplingPointsSig, 
                                                                         samplingDistanceSig, 
                                                                         wavelength, 
                                                                         distance, 
                                                                         doPowerCorrection);
    }
    else {
      signalField = CalculateSignalFieldFromTargetPlaneFieldDiffuserBeamShaper(targetPlaneField,
                                                                               samplingPointsSig, 
                                                                               samplingDistanceSig, 
                                                                               wavelength, 
                                                                               distance, 
                                                                               doPowerCorrection);
    }

    return signalField;
  }

  /// <summary>
  /// [NI] Calculate approximate signal field for beam splitter design from field given in target plane.
  /// Each non-zero sampling value in both fields corresponds to a signal order of the beam splitter.
  /// </summary>
  /// <param name="targetPlaneField">Field in target plane</param>
  /// <param name="samplingPointsSig">Number of sampling points for signal field</param>
  /// <param name="samplingDistanceSig">Sampling distance for signal field in (1/m, 1/m)</param>
  /// <param name="wavelength">Wavelength to compute wave number vectors from signal field points</param>
  /// <param name="distance">Distance to origin point of target plane in (m, m, m)</param>
  /// <param name="doPowerCorrection">If true the power of the orders with higher angles will be increase to compesate an intensity reduction 
  /// because of an increasing of the laser beam diameter.</param>
  /// <returns>Aproximate signal field</returns>
  public static ComplexAmplitude CalculateSignalFieldFromTargetPlaneFieldBeamSplitter(ComplexAmplitude targetPlaneField,
                                                                                      Vector samplingPointsSig,
                                                                                      VectorD samplingDistanceSig,
                                                                                      double wavelength,
                                                                                      double distance,
                                                                                      bool doPowerCorrection) {

    // create signal field
    ComplexAmplitude signalField = new ComplexAmplitude(samplingPointsSig);
    signalField.ComplexAmplitudeType = ComplexAmplitudeType.GloballyPolSpectralComplexAmplitudeField;
    signalField.SamplingDistance = samplingDistanceSig;

    VectorD signalPos;
    double powerCorrectionFactor = 1.0;
    Complex complexValue;

    double x, y;
    int kxPix, kyPix;

    for (int yPix = 0; yPix < targetPlaneField.SamplingPoints.Y; yPix++) {
      y = (yPix - targetPlaneField.SamplingPoints.Y / 2) * targetPlaneField.SamplingDistance.Y;
      for (int xPix = 0; xPix < targetPlaneField.SamplingPoints.X; xPix++) { 
        // iterate through all points of target plane field
        // position in signal plane (pixel coordinates)
        complexValue = targetPlaneField.Field[xPix, yPix];
        if (complexValue != 0.0) {
          x = (xPix - targetPlaneField.SamplingPoints.X / 2) * targetPlaneField.SamplingDistance.X;
          signalPos = GetSignalFieldCoordinates(x, y, wavelength, distance);

          kxPix = (int)(signalPos.X / samplingDistanceSig.X) + samplingPointsSig.X / 2;
          kyPix = (int)(signalPos.Y / samplingDistanceSig.Y) + samplingPointsSig.Y / 2;

          if (doPowerCorrection) {
            double r2 = x * x + y * y + distance * distance;
            powerCorrectionFactor = 1.0 / (distance * distance / r2);
          }
          signalField.Field[kxPix, kyPix] = complexValue * powerCorrectionFactor;
        }
      }
    }
    return signalField;
  }

  /// <summary>
  /// [NI] Calculate approximate signal field for diffuser and beam shaper designs from field given in target plane.
  /// Each non-zero sampling value in both fields corresponds to a signal order of the beam splitter.
  /// </summary>
  /// <param name="targetPlaneField">Field in target plane</param>
  /// <param name="samplingPointsSig">Number of sampling points for signal field</param>
  /// <param name="samplingDistanceSig">Sampling distance for signal field in (1/m, 1/m)</param>
  /// <param name="wavelength">Wavelength to compute wave number vectors from signal field points</param>
  /// <param name="distance">Distance to origin point of target plane in (m, m, m)</param>
  /// <param name="doPowerCorrection">If true the power of the orders with higher angles will be increase to compesate an intensity reduction 
  /// because of an increasing of the laser beam diameter.</param>
  /// <returns>Aproximate signal field</returns>
  public static ComplexAmplitude CalculateSignalFieldFromTargetPlaneFieldDiffuserBeamShaper(ComplexAmplitude targetPlaneField,
                                                                                            Vector samplingPointsSig,
                                                                                            VectorD samplingDistanceSig,
                                                                                            double wavelength,
                                                                                            double distance,
                                                                                            bool doPowerCorrection){
    // create signal field
    ComplexAmplitude signalField = new ComplexAmplitude(samplingPointsSig);
    signalField.ComplexAmplitudeType = ComplexAmplitudeType.GloballyPolSpectralComplexAmplitudeField;
    signalField.SamplingDistance = samplingDistanceSig;

    int ix, iy;
    VectorD coord = new VectorD(0, 0);
    VectorD plane = new VectorD(0, 0);
    double adjustedSignalOrderAmplitude, centralPixelArea = GetPixelArea(0, 0, wavelength, samplingDistanceSig, distance);
    
    double kx, ky;
    double k = MathFunctions.TwoPi / wavelength;

    for (iy = 0; iy < samplingPointsSig.Y; iy++){
      ky = samplingDistanceSig.Y * (iy - samplingPointsSig.Y / 2);
      for (ix = 0; ix < samplingPointsSig.X; ix++){ 
        // iterate through all points of signal field
        // position in target plane (pixel coordinates)
        kx = samplingDistanceSig.X * (ix - samplingPointsSig.X / 2);
        bool noEvanscentWaves = GetTargetPlaneCoordinates(kx, ky, wavelength, distance, out plane);
        if (noEvanscentWaves){
          coord.X = (plane.X / targetPlaneField.SamplingDistance.X) + (targetPlaneField.SamplingPoints.X / 2);
          coord.Y = (plane.Y / targetPlaneField.SamplingDistance.Y) + (targetPlaneField.SamplingPoints.Y / 2);
          if (coord.X > int.MaxValue || coord.X < int.MinValue){
            coord.X = int.MaxValue - 10;
          }
          if (coord.Y > int.MaxValue || coord.Y < int.MinValue){
            coord.Y = int.MaxValue - 10;
          }

          // handle intensity adjustment
          double interpolatedAmplitude = ComplexFieldPointInterpolation.Interpolation(targetPlaneField.Field,
                                                                                      coord,
                                                                                      InterpolationMethod.Linear_AmplitudeAndPhase,
                                                                                      false).Abs();
          if (interpolatedAmplitude != 0){
            double powerCorrectionFactor = 1.0;
            if (doPowerCorrection) {
              double r2 = plane.X * plane.X + plane.Y * plane.Y + distance * distance;
              powerCorrectionFactor = 1.0 / (distance * distance / r2);
            }
            adjustedSignalOrderAmplitude = interpolatedAmplitude * powerCorrectionFactor;
            signalField.Field[ix, iy] = adjustedSignalOrderAmplitude;
          }
          else{
            signalField.Field[ix, iy] = 0;
          }
        }
      }
    }
    return signalField;
  }

  /// <summary>
  /// Calculate target plane coordinates from signal field coordinates. The refractive index is assumed to be one.
  /// </summary>
  /// <param name="kx">Signal field x position (in 1/m)</param>
  /// <param name="ky">Signal field y position (in 1/m)</param>
  /// <param name="plane">Target plane position (in m)</param>
  /// <returns> Returns true if successfull or false of evanescent waves appeared.</returns>
  public static bool GetTargetPlaneCoordinates(double kx,
                                               double ky,
                                               double wavelength,
                                               double distance,
                                               out VectorD plane) {
    plane = new VectorD();

    double k = MathFunctions.TwoPi / wavelength;
    Vector3D planeUnitX = new Vector3D(1, 0, 0);
    Vector3D planeUnitY = new Vector3D(0, 1, 0);
    Vector3D planeNormal = new Vector3D(0, 0, 1);

    // calculate unit wave vector
    double kzSquare = k * k - kx * kx - ky * ky;
    if(kzSquare >= 0) {
      double kz = Math.Sqrt(kzSquare);
      Vector3D unitWaveVector = new Vector3D(kx / k, ky / k, kz / k);

      // calculate beta (| operator is scalar product)
      double beta = distance * planeNormal.Z / (unitWaveVector | planeNormal);
      // position in target plane
      Vector3D temp = unitWaveVector * beta - new Vector3D(0, 0, distance);
      plane.X = temp | planeUnitX;
      plane.Y = temp | planeUnitY;
      return true;
    }
    else {
      plane.X = 0;
      plane.Y = 0;
      return false;
    }
  }

  /// <summary>
  /// Calculate signal field coordinates in space frequencies from target plane field coordinates. The refractive index is assumed to be one.
  /// </summary>
  /// <param name="x">Target plane x position (in m)</param>
  /// <param name="y">Target plane y position (in m)</param>
  /// <returns> Signal field coordinates (in 1/m)</returns>
  public static VectorD GetSignalFieldCoordinates(double x,
                                                  double y,
                                                  double wavelength,
                                                  double distance){
    VectorD signalPos = new VectorD();
    double n = 1.0;

    double r = Math.Sqrt(x * x + y * y + distance * distance);
    double factor = MathFunctions.TwoPi * n / (wavelength * r);
    signalPos.X = factor * x;
    signalPos.Y = factor * y;

    return signalPos;
  }

  /// <summary>
  /// Calculate pixel area in output plane.
  /// Can throw EvanescentWaveNumberException.
  /// </summary>
  /// <param name="kx">Signal field x position</param>
  /// <param name="ky">Signal field y position </param>
  /// <param name="distance"> Distance along z-axis between screen and eye box.</param>
  /// <param name="samplingDistance">Sampling distance of signal field.</param>
  /// <returns>Pixel area in output plane in m^2 if both flags are set to false. If
  /// one of the flags is set to true, a length in m is returned.</returns>
  public static double GetPixelArea(double kx,
                                    double ky,
                                    double wavelength,
                                    VectorD samplingDistance,
                                    double distance) {
    double area = 0;
    // calculate target plane coordinates corresponding to four edges of quadrangle
    VectorD p1 = new VectorD(0, 0), p2 = new VectorD(0, 0), p3 = new VectorD(0, 0), p4 = new VectorD(0, 0);

    bool dummy1, dummy2, dummy3, dummy4;
    dummy1 = GetTargetPlaneCoordinates(kx - 0.5 * samplingDistance.X, ky, wavelength, distance, out p1);
    dummy2 = GetTargetPlaneCoordinates(kx + 0.5 * samplingDistance.X, ky, wavelength, distance, out p3);
    dummy3 = GetTargetPlaneCoordinates(kx, ky - 0.5 * samplingDistance.Y, wavelength, distance, out p2);
    dummy4 = GetTargetPlaneCoordinates(kx, ky + 0.5 * samplingDistance.Y, wavelength, distance, out p4);

    if(dummy1 && dummy2 && dummy3 && dummy4) {
      // calculate area of quadrangle
      VectorD aVec = p1 - p2;
      double a = aVec.Abs();
      VectorD bVec = p2 - p3;
      double b = bVec.Abs();
      VectorD cVec = p3 - p4;
      double c = cVec.Abs();
      VectorD dVec = p4 - p1;
      double d = dVec.Abs();

      double k = (a + b + c + d) / 2;
      double temp = (aVec | dVec) / a / d;
      if(temp > 1) {
        temp = 1;
      }
      double alpha = Math.Acos(temp);
      temp = (bVec | cVec) / b / c;
      if(temp > 1) {
        temp = 1;
      }
      double gamma = Math.Acos(temp);
      double phiCos = Math.Cos((alpha + gamma) / 2);

      area = 2 * Math.Sqrt((k - a) * (k - b) * (k - c) * (k - d) - a * b * c * d * phiCos * phiCos);
    }
    else {
      area = 0;
    }
    return area;
  }
}

/// <summary>
/// public edit form for signal preparation parameters
/// </summary>
public partial class FormSignalPreparationAngularSpectrumDesign : FormDialog {
  #region constructors
  /// <summary>
  /// default constructor
  /// </summary>
  /// <param name="baseCA">the complex amplitude that shall processed</param>
  public FormSignalPreparationAngularSpectrumDesign(ComplexAmplitude baseCA) {
    InitializeComponent();
    this.Wavelength = Globals.DefaultWavelength;
    this.Distance = 1;
    //handling for 1D fields
    if(baseCA.SamplingPoints.X == 1) {
      this.putbFieldSizeX.Visible = false;
      this.nUDSamplingPointsX.Visible = false;
      this.putbSamplingDistanceX.Visible = false;
      this.nUDSamplingPointsX.Minimum = 1;
    }
    if(baseCA.SamplingPoints.Y == 1) {
      this.putbFieldSizeY.Visible = false;
      this.nUDSamplingPointsY.Visible = false;
      this.putbSamplingDistanceY.Visible = false;
      this.nUDSamplingPointsY.Minimum = 1;
    }

    //calculate k-sampling distance (depending on the sampling of the complex amplitude to
    //convert)
    double alpha = Math.Atan2(baseCA.SamplingDistance.X, this.Distance);
    double beta = Math.Atan2(baseCA.SamplingDistance.Y, this.Distance);
    Vector3D unitVector = new Vector3D();
    DirectionConversion.UnitVectorFromCartesianAngles(alpha, beta, out unitVector);
    double k0 = (2 * Math.PI) / this.Wavelength;
    VectorD samplingDistanceK = new VectorD();
    samplingDistanceK.X = unitVector.X * k0;
    samplingDistanceK.Y = unitVector.Y * k0;
    SamplingParameters sParaKSpace = new SamplingParameters();
    sParaKSpace.SamplingPoints = baseCA.SamplingPoints;
    sParaKSpace.SamplingDistance = samplingDistanceK;
    this.SamplingParametersForConversion = sParaKSpace;
  }
  #endregion

  #region properties
  /// <summary>
  /// public property to set and get the wavelength used for conversion
  /// </summary>
  public double Wavelength {
    set {
      putbWavelength.DoubleValue = value;
    }

    get {
      return putbWavelength.DoubleValue;
    }
  }

  /// <summary>
  /// public property to set and get the distance used for conversion to angular spectrum
  /// </summary>
  public double Distance {
    set {
      this.putbDistance.DoubleValue = value;
    }

    get {
      return this.putbDistance.DoubleValue;
    }
  }

  /// <summary>
  /// public property to set and get whether the signal preparation shall be used for beam splitter
  /// or for diffuser/beam shaper
  /// </summary>
  public bool UseForBeamSplitterSignalPreparation {
    get {
      return this.radioButtonSplitter.Checked;
    }

    set {
      this.radioButtonSplitter.Checked = value;
      this.radioButtonDiffuser.Checked = !value;
    }
  }

  /// <summary>
  /// public property to set and get whether the power of the orders shall be corrected depending
  /// on the diffraction angle
  /// </summary>
  public bool CorrectPowerOfOrders {
    get {
      return this.checkBoxCorrection.Checked;
    }

    set {
      this.checkBoxCorrection.Checked = value;
    }
  }

  /// <summary>
  /// public property to get and set the the sampling parameter that shall be used for conversion
  /// </summary>
  public SamplingParameters SamplingParametersForConversion {
    set {
      this.nUDSamplingPointsX.Value = value.SamplingPointsX;
      this.nUDSamplingPointsY.Value = value.SamplingPointsY;
      this.putbSamplingDistanceX.DoubleValue = value.SamplingDistanceX;
      this.putbSamplingDistanceY.DoubleValue = value.SamplingDistanceY;
    }

    get {
      //evaluate the sampling parameters from the UI-elements
      SamplingParameters sParameters = new SamplingParameters();
      sParameters.SamplingPoints = new Vector((int)this.nUDSamplingPointsX.Value, (int)this.nUDSamplingPointsY.Value);
      sParameters.SamplingDistance = new VectorD(this.putbSamplingDistanceX.DoubleValue, this.putbSamplingDistanceY.DoubleValue);

      return sParameters;
    }
  }
  #endregion

  #region methods
  /// <summary>
  /// private support function to update the field size within the GUI
  /// </summary>
  private void updateFieldSize() {
    SamplingParameters sParaCurrent = this.SamplingParametersForConversion;
    this.putbFieldSizeX.DoubleValue = sParaCurrent.Diameter.X;
    this.putbFieldSizeY.DoubleValue = sParaCurrent.Diameter.Y;
  }
  #endregion

  #region events
  /// <summary>
  /// double click handler for sampling points y
  /// </summary>
  /// <param name="sender">the sender of the event</param>
  /// <param name="e">the arguments of the event</param>
  private void nUDSamplingPointsY_DoubleClick(object sender, EventArgs e) {
    if(this.nUDSamplingPointsX.Enabled) {
      this.nUDSamplingPointsY.Value = this.nUDSamplingPointsX.Value;
    }
  }

  /// <summary>
  /// double click handler for sampling points x
  /// </summary>
  /// <param name="sender">the sender of the event</param>
  /// <param name="e">the arguments of the event</param>
  private void nUDSamplingPointsX_DoubleClick(object sender, EventArgs e) {
    if(this.nUDSamplingPointsY.Enabled) {
      this.nUDSamplingPointsX.Value = this.nUDSamplingPointsY.Value;
    }
  }

  /// <summary>
  /// double click handler for sampling distance x
  /// </summary>
  /// <param name="sender">the sender of the event</param>
  private void putbSamplingDistanceX__DoubleClickHandler(object sender) {
    if(this.putbSamplingDistanceY.Visible) {
      this.putbSamplingDistanceX.DoubleValue = this.putbSamplingDistanceY.DoubleValue;
    }
  }

  /// <summary>
  /// double click handler for sampling distance y
  /// </summary>
  /// <param name="sender">the sender of the event</param>
  private void putbSamplingDistanceY__DoubleClickHandler(object sender) {
    if(this.putbSamplingDistanceX.Visible) {
      this.putbSamplingDistanceY.DoubleValue = this.putbSamplingDistanceX.DoubleValue;
    }
  }

  /// <summary>
  /// eventhandling for changings of the sampling points
  /// </summary>
  /// <param name="sender">the sender of the event</param>
  /// <param name="e">the arguments of the event</param>
  private void SamplingPointsChanged(object sender, EventArgs e) {
    updateFieldSize();
  }

  /// <summary>
  /// eventhandling for changings of the sampling distance
  /// </summary>
  /// <param name="sender">the sender of the event</param>
  /// <param name="e">the arguments of the event</param>
  private void SamplingDistanceChanged(object sender, EventArgs e) {
    updateFieldSize();
  }
  #endregion
 
  #region designer code
  /// <summary>
  /// Required designer variable.
  /// </summary>
  private System.ComponentModel.IContainer components = null;

  /// <summary>
  /// Clean up any resources being used.
  /// </summary>
  /// <param name="disposing">true if managed resources should be disposed; otherwise, false.</param>
  protected override void Dispose(bool disposing) {
    if(disposing && (components != null)) {
      components.Dispose();
    }
    base.Dispose(disposing);
  }

  #region Windows Form Designer generated code
  /// <summary>
  /// Required method for Designer support - do not modify
  /// the contents of this method with the code editor.
  /// </summary>
  private void InitializeComponent() {
    this.labelWavelength = new System.Windows.Forms.Label();
    this.putbWavelength = new VirtualLabAPI.UI.BasicUI.PhysicalUnitTextBox();
    this.labelDistance = new System.Windows.Forms.Label();
    this.groupBoxSamplingParameters = new System.Windows.Forms.GroupBox();
    this.labelFieldSize = new System.Windows.Forms.Label();
    this.label2 = new System.Windows.Forms.Label();
    this.putbFieldSizeX = new VirtualLabAPI.UI.BasicUI.PhysicalUnitTextBox();
    this.putbFieldSizeY = new VirtualLabAPI.UI.BasicUI.PhysicalUnitTextBox();
    this.labelSamplingDistance = new System.Windows.Forms.Label();
    this.labelSamplingDistanceX = new System.Windows.Forms.Label();
    this.putbSamplingDistanceX = new VirtualLabAPI.UI.BasicUI.PhysicalUnitTextBox();
    this.putbSamplingDistanceY = new VirtualLabAPI.UI.BasicUI.PhysicalUnitTextBox();
    this.labelSPX = new System.Windows.Forms.Label();
    this.nUDSamplingPointsY = new System.Windows.Forms.NumericUpDown();
    this.nUDSamplingPointsX = new System.Windows.Forms.NumericUpDown();
    this.labelSamplingPoints = new System.Windows.Forms.Label();
    this.putbDistance = new VirtualLabAPI.UI.BasicUI.PhysicalUnitTextBox();
    this.buttonOK = new System.Windows.Forms.Button();
    this.buttonCancel = new System.Windows.Forms.Button();
    this.groupBoxConversionParameters = new System.Windows.Forms.GroupBox();
    this.checkBoxCorrection = new System.Windows.Forms.CheckBox();
    this.radioButtonDiffuser = new System.Windows.Forms.RadioButton();
    this.radioButtonSplitter = new System.Windows.Forms.RadioButton();
    this.labelType = new System.Windows.Forms.Label();
    this.groupBoxSamplingParameters.SuspendLayout();
    ((System.ComponentModel.ISupportInitialize)(this.nUDSamplingPointsY)).BeginInit();
    ((System.ComponentModel.ISupportInitialize)(this.nUDSamplingPointsX)).BeginInit();
    this.groupBoxConversionParameters.SuspendLayout();
    this.SuspendLayout();
    // 
    // labelWavelength
    // 
    this.labelWavelength.AutoSize = true;
    this.labelWavelength.Location = new System.Drawing.Point(12, 20);
    this.labelWavelength.Name = "labelWavelength";
    this.labelWavelength.Size = new System.Drawing.Size(65, 13);
    this.labelWavelength.TabIndex = 0;
    this.labelWavelength.Text = "Wavelength";
    // 
    // putbWavelength
    // 
    this.putbWavelength.CausesValidation = false;
    this.putbWavelength.Location = new System.Drawing.Point(311, 17);
    this.putbWavelength.MaximumSize = new System.Drawing.Size(300, 24);
    this.putbWavelength.MinimumDouble = 0D;
    this.putbWavelength.MinimumInteger = 0;
    this.putbWavelength.MinimumSize = new System.Drawing.Size(0, 24);
    this.putbWavelength.MinimumValueIsAllowed = false;
    this.putbWavelength.Name = "putbWavelength";
    this.putbWavelength.Size = new System.Drawing.Size(117, 24);
    this.putbWavelength.TabIndex = 0;
    this.putbWavelength.TextboxBackColor = System.Drawing.SystemColors.Window;
    // 
    // labelDistance
    // 
    this.labelDistance.AutoSize = true;
    this.labelDistance.Location = new System.Drawing.Point(13, 25);
    this.labelDistance.Name = "labelDistance";
    this.labelDistance.Size = new System.Drawing.Size(49, 13);
    this.labelDistance.TabIndex = 0;
    this.labelDistance.Text = "Distance";
    // 
    // groupBoxSamplingParameters
    // 
    this.groupBoxSamplingParameters.Controls.Add(this.labelFieldSize);
    this.groupBoxSamplingParameters.Controls.Add(this.label2);
    this.groupBoxSamplingParameters.Controls.Add(this.putbFieldSizeX);
    this.groupBoxSamplingParameters.Controls.Add(this.putbFieldSizeY);
    this.groupBoxSamplingParameters.Controls.Add(this.labelSamplingDistance);
    this.groupBoxSamplingParameters.Controls.Add(this.labelSamplingDistanceX);
    this.groupBoxSamplingParameters.Controls.Add(this.putbSamplingDistanceX);
    this.groupBoxSamplingParameters.Controls.Add(this.putbSamplingDistanceY);
    this.groupBoxSamplingParameters.Controls.Add(this.labelSPX);
    this.groupBoxSamplingParameters.Controls.Add(this.nUDSamplingPointsY);
    this.groupBoxSamplingParameters.Controls.Add(this.nUDSamplingPointsX);
    this.groupBoxSamplingParameters.Controls.Add(this.labelSamplingPoints);
    this.groupBoxSamplingParameters.Location = new System.Drawing.Point(12, 165);
    this.groupBoxSamplingParameters.Name = "groupBoxSamplingParameters";
    this.groupBoxSamplingParameters.Size = new System.Drawing.Size(430, 125);
    this.groupBoxSamplingParameters.TabIndex = 2;
    this.groupBoxSamplingParameters.TabStop = false;
    this.groupBoxSamplingParameters.Text = "Sampling Parameters";
    // 
    // labelFieldSize
    // 
    this.labelFieldSize.AutoSize = true;
    this.labelFieldSize.Enabled = false;
    this.labelFieldSize.Location = new System.Drawing.Point(6, 90);
    this.labelFieldSize.Name = "labelFieldSize";
    this.labelFieldSize.Size = new System.Drawing.Size(52, 13);
    this.labelFieldSize.TabIndex = 8;
    this.labelFieldSize.Text = "Field Size";
    // 
    // label2
    // 
    this.label2.AutoSize = true;
    this.label2.Enabled = false;
    this.label2.Location = new System.Drawing.Point(271, 90);
    this.label2.Name = "label2";
    this.label2.Size = new System.Drawing.Size(12, 13);
    this.label2.TabIndex = 10;
    this.label2.Text = "x";
    // 
    // putbFieldSizeX
    // 
    this.putbFieldSizeX.CausesValidation = false;
    this.putbFieldSizeX.Enabled = false;
    this.putbFieldSizeX.Location = new System.Drawing.Point(145, 87);
    this.putbFieldSizeX.MaximumSize = new System.Drawing.Size(300, 24);
    this.putbFieldSizeX.MinimumDouble = 0D;
    this.putbFieldSizeX.MinimumInteger = 0;
    this.putbFieldSizeX.MinimumSize = new System.Drawing.Size(0, 24);
    this.putbFieldSizeX.MinimumValueIsAllowed = false;
    this.putbFieldSizeX.Name = "putbFieldSizeX";
    this.putbFieldSizeX.PhysicalProperty = VirtualLabAPI.Core.Numerics.PhysicalProperty.WaveNumber;
    this.putbFieldSizeX.Size = new System.Drawing.Size(120, 24);
    this.putbFieldSizeX.TabIndex = 9;
    this.putbFieldSizeX.TextboxBackColor = System.Drawing.SystemColors.Window;
    // 
    // putbFieldSizeY
    // 
    this.putbFieldSizeY.CausesValidation = false;
    this.putbFieldSizeY.Enabled = false;
    this.putbFieldSizeY.Location = new System.Drawing.Point(296, 87);
    this.putbFieldSizeY.MaximumSize = new System.Drawing.Size(300, 24);
    this.putbFieldSizeY.MinimumDouble = 0D;
    this.putbFieldSizeY.MinimumInteger = 0;
    this.putbFieldSizeY.MinimumSize = new System.Drawing.Size(0, 24);
    this.putbFieldSizeY.MinimumValueIsAllowed = false;
    this.putbFieldSizeY.Name = "putbFieldSizeY";
    this.putbFieldSizeY.PhysicalProperty = VirtualLabAPI.Core.Numerics.PhysicalProperty.WaveNumber;
    this.putbFieldSizeY.Size = new System.Drawing.Size(120, 24);
    this.putbFieldSizeY.TabIndex = 11;
    this.putbFieldSizeY.TextboxBackColor = System.Drawing.SystemColors.Window;
    // 
    // labelSamplingDistance
    // 
    this.labelSamplingDistance.AutoSize = true;
    this.labelSamplingDistance.Location = new System.Drawing.Point(6, 57);
    this.labelSamplingDistance.Name = "labelSamplingDistance";
    this.labelSamplingDistance.Size = new System.Drawing.Size(95, 13);
    this.labelSamplingDistance.TabIndex = 4;
    this.labelSamplingDistance.Text = "Sampling Distance";
    // 
    // labelSamplingDistanceX
    // 
    this.labelSamplingDistanceX.AutoSize = true;
    this.labelSamplingDistanceX.Location = new System.Drawing.Point(271, 57);
    this.labelSamplingDistanceX.Name = "labelSamplingDistanceX";
    this.labelSamplingDistanceX.Size = new System.Drawing.Size(12, 13);
    this.labelSamplingDistanceX.TabIndex = 6;
    this.labelSamplingDistanceX.Text = "x";
    // 
    // putbSamplingDistanceX
    // 
    this.putbSamplingDistanceX.CausesValidation = false;
    this.putbSamplingDistanceX.Location = new System.Drawing.Point(145, 54);
    this.putbSamplingDistanceX.MaximumSize = new System.Drawing.Size(300, 24);
    this.putbSamplingDistanceX.MinimumDouble = 0D;
    this.putbSamplingDistanceX.MinimumInteger = 0;
    this.putbSamplingDistanceX.MinimumSize = new System.Drawing.Size(0, 24);
    this.putbSamplingDistanceX.MinimumValueIsAllowed = false;
    this.putbSamplingDistanceX.Name = "putbSamplingDistanceX";
    this.putbSamplingDistanceX.PhysicalProperty = VirtualLabAPI.Core.Numerics.PhysicalProperty.WaveNumber;
    this.putbSamplingDistanceX.Size = new System.Drawing.Size(120, 24);
    this.putbSamplingDistanceX.TabIndex = 5;
    this.putbSamplingDistanceX.TextboxBackColor = System.Drawing.SystemColors.Window;
    this.putbSamplingDistanceX.ValueChanged += new System.EventHandler(this.SamplingDistanceChanged);
    this.putbSamplingDistanceX._DoubleClickHandler += new VirtualLabAPI.UI.BasicUI.PhysicalUnitTextBox.BoxMouseDoubleClickHandler(this.putbSamplingDistanceX__DoubleClickHandler);
    // 
    // putbSamplingDistanceY
    // 
    this.putbSamplingDistanceY.CausesValidation = false;
    this.putbSamplingDistanceY.Location = new System.Drawing.Point(296, 54);
    this.putbSamplingDistanceY.MaximumSize = new System.Drawing.Size(300, 24);
    this.putbSamplingDistanceY.MinimumDouble = 0D;
    this.putbSamplingDistanceY.MinimumInteger = 0;
    this.putbSamplingDistanceY.MinimumSize = new System.Drawing.Size(0, 24);
    this.putbSamplingDistanceY.MinimumValueIsAllowed = false;
    this.putbSamplingDistanceY.Name = "putbSamplingDistanceY";
    this.putbSamplingDistanceY.PhysicalProperty = VirtualLabAPI.Core.Numerics.PhysicalProperty.WaveNumber;
    this.putbSamplingDistanceY.Size = new System.Drawing.Size(120, 24);
    this.putbSamplingDistanceY.TabIndex = 7;
    this.putbSamplingDistanceY.TextboxBackColor = System.Drawing.SystemColors.Window;
    this.putbSamplingDistanceY.ValueChanged += new System.EventHandler(this.SamplingDistanceChanged);
    this.putbSamplingDistanceY._DoubleClickHandler += new VirtualLabAPI.UI.BasicUI.PhysicalUnitTextBox.BoxMouseDoubleClickHandler(this.putbSamplingDistanceY__DoubleClickHandler);
    // 
    // labelSPX
    // 
    this.labelSPX.AutoSize = true;
    this.labelSPX.Location = new System.Drawing.Point(271, 25);
    this.labelSPX.Name = "labelSPX";
    this.labelSPX.Size = new System.Drawing.Size(12, 13);
    this.labelSPX.TabIndex = 2;
    this.labelSPX.Text = "x";
    // 
    // nUDSamplingPointsY
    // 
    this.nUDSamplingPointsY.Location = new System.Drawing.Point(316, 23);
    this.nUDSamplingPointsY.Maximum = new decimal(new int[] {
            20000,
            0,
            0,
            0});
    this.nUDSamplingPointsY.Minimum = new decimal(new int[] {
            2,
            0,
            0,
            0});
    this.nUDSamplingPointsY.Name = "nUDSamplingPointsY";
    this.nUDSamplingPointsY.Size = new System.Drawing.Size(100, 20);
    this.nUDSamplingPointsY.TabIndex = 3;
    this.nUDSamplingPointsY.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
    this.nUDSamplingPointsY.Value = new decimal(new int[] {
            2,
            0,
            0,
            0});
    this.nUDSamplingPointsY.ValueChanged += new System.EventHandler(this.SamplingPointsChanged);
    this.nUDSamplingPointsY.DoubleClick += new System.EventHandler(this.nUDSamplingPointsY_DoubleClick);
    // 
    // nUDSamplingPointsX
    // 
    this.nUDSamplingPointsX.Location = new System.Drawing.Point(165, 23);
    this.nUDSamplingPointsX.Maximum = new decimal(new int[] {
            20000,
            0,
            0,
            0});
    this.nUDSamplingPointsX.Minimum = new decimal(new int[] {
            2,
            0,
            0,
            0});
    this.nUDSamplingPointsX.Name = "nUDSamplingPointsX";
    this.nUDSamplingPointsX.Size = new System.Drawing.Size(100, 20);
    this.nUDSamplingPointsX.TabIndex = 1;
    this.nUDSamplingPointsX.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
    this.nUDSamplingPointsX.Value = new decimal(new int[] {
            2,
            0,
            0,
            0});
    this.nUDSamplingPointsX.ValueChanged += new System.EventHandler(this.SamplingPointsChanged);
    this.nUDSamplingPointsX.DoubleClick += new System.EventHandler(this.nUDSamplingPointsX_DoubleClick);
    // 
    // labelSamplingPoints
    // 
    this.labelSamplingPoints.AutoSize = true;
    this.labelSamplingPoints.Location = new System.Drawing.Point(6, 25);
    this.labelSamplingPoints.Name = "labelSamplingPoints";
    this.labelSamplingPoints.Size = new System.Drawing.Size(82, 13);
    this.labelSamplingPoints.TabIndex = 0;
    this.labelSamplingPoints.Text = "Sampling Points";
    // 
    // putbDistance
    // 
    this.putbDistance.CausesValidation = false;
    this.putbDistance.Location = new System.Drawing.Point(299, 22);
    this.putbDistance.MaximumSize = new System.Drawing.Size(300, 24);
    this.putbDistance.MinimumDouble = 0D;
    this.putbDistance.MinimumInteger = 0;
    this.putbDistance.MinimumSize = new System.Drawing.Size(0, 24);
    this.putbDistance.MinimumValueIsAllowed = false;
    this.putbDistance.Name = "putbDistance";
    this.putbDistance.Size = new System.Drawing.Size(117, 24);
    this.putbDistance.TabIndex = 1;
    this.putbDistance.TextboxBackColor = System.Drawing.SystemColors.Window;
    // 
    // buttonOK
    // 
    this.buttonOK.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Right)));
    this.buttonOK.DialogResult = System.Windows.Forms.DialogResult.OK;
    this.buttonOK.Location = new System.Drawing.Point(274, 296);
    this.buttonOK.Name = "buttonOK";
    this.buttonOK.Size = new System.Drawing.Size(75, 23);
    this.buttonOK.TabIndex = 3;
    this.buttonOK.Text = "&OK";
    this.buttonOK.UseVisualStyleBackColor = true;
    // 
    // buttonCancel
    // 
    this.buttonCancel.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Right)));
    this.buttonCancel.DialogResult = System.Windows.Forms.DialogResult.Cancel;
    this.buttonCancel.Location = new System.Drawing.Point(364, 296);
    this.buttonCancel.Name = "buttonCancel";
    this.buttonCancel.Size = new System.Drawing.Size(75, 23);
    this.buttonCancel.TabIndex = 4;
    this.buttonCancel.Text = "&Cancel";
    this.buttonCancel.UseVisualStyleBackColor = true;
    // 
    // groupBoxConversionParameters
    // 
    this.groupBoxConversionParameters.Controls.Add(this.checkBoxCorrection);
    this.groupBoxConversionParameters.Controls.Add(this.radioButtonDiffuser);
    this.groupBoxConversionParameters.Controls.Add(this.radioButtonSplitter);
    this.groupBoxConversionParameters.Controls.Add(this.labelType);
    this.groupBoxConversionParameters.Controls.Add(this.labelDistance);
    this.groupBoxConversionParameters.Controls.Add(this.putbDistance);
    this.groupBoxConversionParameters.Location = new System.Drawing.Point(12, 43);
    this.groupBoxConversionParameters.Name = "groupBoxConversionParameters";
    this.groupBoxConversionParameters.Size = new System.Drawing.Size(430, 105);
    this.groupBoxConversionParameters.TabIndex = 1;
    this.groupBoxConversionParameters.TabStop = false;
    this.groupBoxConversionParameters.Text = "Conversion Parameters";
    // 
    // checkBoxCorrection
    // 
    this.checkBoxCorrection.AutoSize = true;
    this.checkBoxCorrection.Checked = true;
    this.checkBoxCorrection.CheckState = System.Windows.Forms.CheckState.Checked;
    this.checkBoxCorrection.Location = new System.Drawing.Point(16, 81);
    this.checkBoxCorrection.Name = "checkBoxCorrection";
    this.checkBoxCorrection.Size = new System.Drawing.Size(283, 17);
    this.checkBoxCorrection.TabIndex = 5;
    this.checkBoxCorrection.Text = "Correct Power of Orders According to Diffraction Angle";
    this.checkBoxCorrection.UseVisualStyleBackColor = true;
    // 
    // radioButtonDiffuser
    // 
    this.radioButtonDiffuser.AutoSize = true;
    this.radioButtonDiffuser.Location = new System.Drawing.Point(276, 51);
    this.radioButtonDiffuser.Name = "radioButtonDiffuser";
    this.radioButtonDiffuser.Size = new System.Drawing.Size(140, 17);
    this.radioButtonDiffuser.TabIndex = 4;
    this.radioButtonDiffuser.Text = "Diffuser or Beam Shaper";
    this.radioButtonDiffuser.UseVisualStyleBackColor = true;
    // 
    // radioButtonSplitter
    // 
    this.radioButtonSplitter.AutoSize = true;
    this.radioButtonSplitter.Checked = true;
    this.radioButtonSplitter.Location = new System.Drawing.Point(165, 51);
    this.radioButtonSplitter.Name = "radioButtonSplitter";
    this.radioButtonSplitter.Size = new System.Drawing.Size(87, 17);
    this.radioButtonSplitter.TabIndex = 3;
    this.radioButtonSplitter.TabStop = true;
    this.radioButtonSplitter.Text = "Beam Splitter";
    this.radioButtonSplitter.UseVisualStyleBackColor = true;
    // 
    // labelType
    // 
    this.labelType.AutoSize = true;
    this.labelType.Location = new System.Drawing.Point(13, 53);
    this.labelType.Name = "labelType";
    this.labelType.Size = new System.Drawing.Size(140, 13);
    this.labelType.TabIndex = 2;
    this.labelType.Text = "Calculation of Signal Field of";
    // 
    // FormSignalPreparationAngularSpectrumDesign
    // 
    this.AcceptButton = this.buttonOK;
    this.AutoScaleDimensions = new System.Drawing.SizeF(6F, 13F);
    this.AutoScaleMode = System.Windows.Forms.AutoScaleMode.Font;
    this.CancelButton = this.buttonCancel;
    this.ClientSize = new System.Drawing.Size(451, 331);
    this.Controls.Add(this.groupBoxConversionParameters);
    this.Controls.Add(this.buttonCancel);
    this.Controls.Add(this.buttonOK);
    this.Controls.Add(this.groupBoxSamplingParameters);
    this.Controls.Add(this.putbWavelength);
    this.Controls.Add(this.labelWavelength);
    this.Font = new System.Drawing.Font("Microsoft Sans Serif", 8.25F);
    this.Name = "FormSignalPreparationAngularSpectrumDesign";
    this.Text = "Conversion Parameters";
    this.groupBoxSamplingParameters.ResumeLayout(false);
    this.groupBoxSamplingParameters.PerformLayout();
    ((System.ComponentModel.ISupportInitialize)(this.nUDSamplingPointsY)).EndInit();
    ((System.ComponentModel.ISupportInitialize)(this.nUDSamplingPointsX)).EndInit();
    this.groupBoxConversionParameters.ResumeLayout(false);
    this.groupBoxConversionParameters.PerformLayout();
    this.ResumeLayout(false);
    this.PerformLayout();
  }
  #endregion

  private System.Windows.Forms.Label labelWavelength;
  private VirtualLabAPI.UI.BasicUI.PhysicalUnitTextBox putbWavelength;
  private System.Windows.Forms.Label labelDistance;
  private System.Windows.Forms.GroupBox groupBoxSamplingParameters;
  private System.Windows.Forms.Label labelSPX;
  private System.Windows.Forms.NumericUpDown nUDSamplingPointsY;
  private System.Windows.Forms.NumericUpDown nUDSamplingPointsX;
  private System.Windows.Forms.Label labelSamplingPoints;
  private VirtualLabAPI.UI.BasicUI.PhysicalUnitTextBox putbDistance;
  private System.Windows.Forms.Button buttonOK;
  private System.Windows.Forms.Button buttonCancel;
  private System.Windows.Forms.Label labelFieldSize;
  private System.Windows.Forms.Label label2;
  private VirtualLabAPI.UI.BasicUI.PhysicalUnitTextBox putbFieldSizeX;
  private VirtualLabAPI.UI.BasicUI.PhysicalUnitTextBox putbFieldSizeY;
  private System.Windows.Forms.Label labelSamplingDistance;
  private System.Windows.Forms.Label labelSamplingDistanceX;
  private VirtualLabAPI.UI.BasicUI.PhysicalUnitTextBox putbSamplingDistanceX;
  private VirtualLabAPI.UI.BasicUI.PhysicalUnitTextBox putbSamplingDistanceY;
  private System.Windows.Forms.GroupBox groupBoxConversionParameters;
  private System.Windows.Forms.CheckBox checkBoxCorrection;
  private System.Windows.Forms.RadioButton radioButtonDiffuser;
  private System.Windows.Forms.RadioButton radioButtonSplitter;
  private System.Windows.Forms.Label labelType;
  #endregion
}